import { useEffect, useState } from 'react'
import { Swiper, SwiperSlide } from 'swiper/react'
import {
  Navigation,
  Pagination,
  Mousewheel,
  Keyboard,
  EffectCoverflow,
} from 'swiper/modules'
import 'swiper/css'
import 'swiper/css/navigation'
import 'swiper/css/pagination'
import 'swiper/css/effect-coverflow'
import { IMAGES_URL } from '@/constants'
import arrow from 'imgs/arrow.png'
import { useMediaQuery } from 'react-responsive'

interface Slider {
  title: string
  text: string
  btnText: string
  carousel_images: {
    data: {
      attributes: {
        url: string
      }
    }[]
  }
}

interface CarouselProps {
  sliders: Slider[]
  mainTitle: string
  backGroundImage: { data: { attributes: { url: string } }[] }
}

const Carousel: React.FC<CarouselProps> = ({
  sliders,
  mainTitle,
  backGroundImage,
}) => {
  const [filteredSliders, setFilteredSliders] = useState<Slider[]>([])
  const [currentSlide, setCurrentSlide] = useState(0)
  const [swiperInstance, setSwiperInstance] = useState<any>(null)
  const isDesktopOrLaptop = useMediaQuery({ query: '(min-width: 1024px)' })
  const isMobile = useMediaQuery({ query: '(max-width: 1023px)' })

  const backgroundImageUrl = isDesktopOrLaptop
    ? IMAGES_URL + backGroundImage.data[0].attributes.url
    : IMAGES_URL + backGroundImage.data[1].attributes.url

  function formatString(input: string) {
    return input.toLowerCase().replace(/\s+/g, '')
  }

  useEffect(() => {
    const filterSliders = async () => {
      const validSliders = await Promise.all(
        sliders.map(async (slider, index) => {
          const response = await fetch(`/service/${formatString(slider.title)}`)
          if (response.ok) {
            return slider
          } else {
            return null
          }
        })
      )
      setFilteredSliders(
        validSliders.filter((slider) => slider !== null) as Slider[]
      )
    }

    filterSliders()
  }, [sliders])



  useEffect(() => {
    if (swiperInstance) {
      swiperInstance.slideTo(0); // Asegura que el carrusel comience en la primera slide
    }
  }, [swiperInstance]);
  

  const handleRedirect = async (title: string) => {
    const response = await fetch(`/service/${title}`)
    if (response.ok) {
      window.location.href = `/service/${title}`
    } else {
      console.error('Route does not exist')
    }
  }

  /* const calculateHeight = (title: string) => {
    const lineHeight = 1.2
    const baseFontSize = isDesktopOrLaptop ? 50 / 16 : 28 / 16
    const lines = Math.ceil(title.length / 20)
    return lineHeight * baseFontSize * lines + 'em'
  } */

  return (
    <div className='overflow-hidden relative '>
      <div className='relative flex pl-[16px] items-center md:pl-[160px]'>
        <div className='bg-gradient-to-t from-azulGradiente from-0% to-verdeGradiente to-100% w-[8px] shrink-0 mt-[6px] md:mt-[12px] h-auto py-[11px] md:py-[20px]' />
        <h2 className='text-[28px] pl-4 font-bold uppercase md:text-[50px] text-white'>
          {mainTitle}
        </h2>
      </div>

      <div
        className='relative w-full min-h-screen mx-auto overflow-hidden'
        style={{
          backgroundImage: `url(${backgroundImageUrl})`,
          backgroundSize: isMobile ? 'contain' : 'cover',
          backgroundPosition: 'center',
        }}
      >
        <Swiper
          effect='coverflow'
          grabCursor={true}
          centeredSlides={true}
          slidesPerView={1.5}
          coverflowEffect={{
            rotate: 0,
            stretch: -550,
            depth: 800,
            modifier: 1,
            slideShadows: false,
          }}
          navigation={{
            nextEl: isDesktopOrLaptop ? '.swiper-button-next' : undefined,
            prevEl: isDesktopOrLaptop ? '.swiper-button-prev' : undefined,
          }}
          pagination={{
            clickable: true,
            el: '.swiper-pagination-custom',
            bulletClass: 'swiper-pagination-custom-bullet',
            bulletActiveClass: 'swiper-pagination-custom-bullet-active',
          }}
          mousewheel={true}
          keyboard={false}
          modules={[
            Navigation,
            Pagination,
            Mousewheel,
            Keyboard,
            EffectCoverflow,
          ]}
          className='mySwiper'
          loop={false}
          breakpoints={{
            1440: {
              slidesPerView: 1.5,
              spaceBetween: 300,
              coverflowEffect: {
                rotate: 0,
                stretch: -550,
                depth: 1100,
                modifier: 1,
                slideShadows: false,
              },
            },
            1920: {
              slidesPerView: 2,
              spaceBetween: 200,
              coverflowEffect: {
                rotate: 0,
                stretch: -850,
                depth: 800,
                modifier: 1,
                slideShadows: false,
              },
            },
          }}
          style={{ overflow: 'hidden' }}
          onSwiper={(swiper) => setSwiperInstance(swiper)}
          initialSlide={0} 
        >
          {filteredSliders.map((slider, index) => (
            <SwiperSlide
              key={index}
              className={`bg-transparent w-[100%] ${
                index === 0 ? 'swiper-slide-first' : ''
              } ${
                index === filteredSliders.length - 1 ? 'swiper-slide-last' : ''
              }`}
            >
              <div className='flex flex-col items-center justify-center mb-[100px] 2xl:mb-[100px] xl:mb-[100px] text-center space-y-4 md:mt-[163px] md:flex-row md:items-center md:space-y-0 md:space-x-8 lg:flex-row lg:items-center lg:space-y-0 lg:space-x-8 lg:mt-[163px]'>
                <div className='relative mt-[50px] flex-shrink-0 md:w-[379px] md:h-[379px] lg:w-[250px] lg:h-[250px] lg:ml-[-100px] xl:mb-[165px]'>
                  <img
                    src={
                      IMAGES_URL + slider.carousel_images.data[0].attributes.url
                    }
                    alt={`Icon ${index + 1}`}
                    className='w-full h-full lg:mt-[0px]'
                  />
                  {slider.carousel_images.data[1] && (
                    <div className='absolute inset-0 flex justify-center items-center'>
                      <img
                        src={
                          IMAGES_URL +
                          slider.carousel_images.data[1].attributes.url
                        }
                        alt={`Overlay ${index + 1}`}
                        className='w-[90px] h-[91px] md:w-[175px] md:h-[175px] lg:w-[120px] lg:h-[120px]'
                      />
                    </div>
                  )}
                </div>
                <div className='flex flex-col items-left text-left lg:mt-[163px] md:items-start md:text-left md:flex-1 mb-[50px]'>
                  <h2 className='text-[20px] mb-4 font-bold text-white md:text-[50px] md:mt-[20px] lg:text-[40px]'>
                    {slider.title}
                  </h2>
                  <p className='text-[16px] text-white md:text-[28px] md:max-w-[800px] lg:max-w-[800px]'>
                    {slider.text}
                  </p>
                  <button
                    onClick={() => handleRedirect(formatString(slider.title))}
                    className='bg-white text-left font-semibold  mr-[60px] md:mr-[0px] text-[#0F7D87] w-[78%] md:w-[261px] mt-[32px] md:mt-[32px] text-[16px] md:text-[24px] px-4 py-2 rounded-full'
                  >
                    {slider.btnText}
                  </button>
                </div>
              </div>
            </SwiperSlide>
          ))}
        </Swiper>
        <div className='swiper-pagination-custom flex justify-center relative items-center w-full'></div>
        {isDesktopOrLaptop && (
          <>
            <div className='swiper-button-prev absolute bg-transparent left-[20px] md:left-[200px] md:ml-[200px] xl:left-[10%] xl:ml-[5%] 2xl:left-[50px] 2xl:ml-[100px]'>
              <img
                src={arrow.src}
                alt='Previous arrow'
                className='w-6 h-6 rotate-180 md:w-[48px] md:h-[48px]'
              />
            </div>
            <div className='swiper-button-next absolute right-[20px] md:right-[0px] md:mr-[200px] xl:right-[10%] xl:mr-[5%] 2xl:right-[50px] 2xl:mr-[100px]'>
              <img
                src={arrow.src}
                alt='Next arrow'
                className='w-6 h-6 md:w-[48px] md:h-[48px]'
              />
            </div>
          </>
        )}
      </div>
    </div>
  )
}

export default Carousel
