import { useState, useEffect, useRef } from 'react';
import anime from 'animejs';
import { IMAGES_URL } from '@/constants';
import { useMediaQuery } from 'react-responsive';

const InteractiveComponent: React.FC<{ attributes: any }> = ({ attributes }) => {
  const circlesNosotros = attributes.Circle_Nosotros || [];
  const [centerTitle, setCenterTitle] = useState<string | null>(null);
  const [centerText, setCenterText] = useState<string[] | null>(null); 
  const [centerImage, setCenterImage] = useState<string | null>(null);
  const [hoveredIndex, setHoveredIndex] = useState<number | null>(null);

  const titles = circlesNosotros.map((circle: any) => circle.title).reverse();
  const images = circlesNosotros.map((circle: any) => IMAGES_URL + circle.Values_image.data.attributes.url).reverse();
  const animationRef = useRef<anime.AnimeInstance | null>(null);

  useEffect(() => {
    const animation = anime({
      targets: '.animated-circle',
      r: [
        { value: 23, duration: 1500 },
        { value: 15, duration: 1500 }
      ],
      strokeWidth: [
        { value: 3, duration: 1500 },
        { value: 6, duration: 1500 }
      ],
      easing: 'easeInOutQuad',
      loop: true,
      autoplay: true
    });

    animationRef.current = animation;
  }, []);

  const is1440px = useMediaQuery({ query: '(min-width: 1024px) and (max-width: 1440px)' });

  const handleMouseEnter = (index: number) => {
    setCenterTitle(titles[index]);
    setCenterImage(images[index]); 
    setHoveredIndex(index);
    if (animationRef.current) {
      animationRef.current.pause();
      const circles = document.querySelectorAll('.animated-circle');
      circles.forEach((circle, i) => {
        if (i !== index) {
          circle.setAttribute('r', '15');
          circle.setAttribute('stroke-width', '6');
        }
      });
    }
  };

  const handleMouseLeave = () => {
    setCenterTitle(null);
    setCenterText(null);
    setCenterImage(null); 
    setHoveredIndex(null);
    if (animationRef.current) {
      animationRef.current.play();
    }
  };

  const smallCircles = titles.map((title, index) => {
    const angle = (index * 360) / 10;
    const radius = 420;
    const x = 533.685 + radius * Math.cos((angle * Math.PI) / 180);
    const y = 533.685 + radius * Math.sin((angle * Math.PI) / 180);

    return (
      <circle
        key={index}
        cx={x}
        cy={y}
        r={hoveredIndex === index ? 23 : 15}
        fill="url(#gradient)"
        stroke="white"
        strokeWidth={hoveredIndex === index ? 3 : 6}
        className={`cursor-pointer ${hoveredIndex !== index ? 'animated-circle' : ''}`}
        onMouseEnter={() => handleMouseEnter(index)}
        onMouseLeave={handleMouseLeave}
      />
    );
  });


const splitTextToLines = (text: string): string[] => {
  const words = text.split(' ');
  const lines = [];
  for (let i = 0; i < words.length; i += 2) {
    lines.push(words.slice(i, i + 2).join(' '));
  }
  return lines;
};

  return (
    <div className="relative flex justify-center items-center h-screen bg-gradient-to-t from-black to-transparent">
      <svg
        width="100%"
        height="100%"
        viewBox="0 0 1067.37 1067.37"
        className="absolute top-[-300px] 2xl:top-[-290px] xl:top-[-240px]"
        preserveAspectRatio="xMidYMid meet"
      >
        <defs>
          <linearGradient id="gradient" x1="0%" y1="0%" x2="100%">
            <stop offset="0%" style={{ stopColor: '#0F7D87', stopOpacity: 1 }} />
            <stop offset="100%" style={{ stopColor: '#86B22C', stopOpacity: 1 }} />
          </linearGradient>
        </defs>

        <circle cx="533.685" cy="533.685" r="340" fill="white" stroke="url(#gradient)" strokeWidth="60" />
        {centerTitle ? (
          <>
            <g transform="translate(533.685, 533.685)">
              <image 
                href={centerImage} 
                x={is1440px ? "-120" : "-125"} 
                y={is1440px ? "-190" : "-190"} 
                width={is1440px ? "250" : "250"} 
                height={is1440px ? "250" : "250"} 
              />
              <text
                x="0"
                y={is1440px ? "70" : "70"}
                textAnchor="middle"
                dominantBaseline="middle"
                fill="gray"
                fontSize="50"
                fontWeight="light"
              >
                {splitTextToLines(centerTitle).map((line, i) => (
                 <tspan key={i} x="0" dy={i === 0 ? "1em" : "1.2em"}>
                    {line}
                  </tspan>
                ))}
              </text>
            </g>
          </>
        ) : (
          <image
            href="/Valores.svg"
            width="560"
            height="500"
            x="253"
            y="260"
          />
        )}

        <circle cx="533.685" cy="533.685" r="390" fill="transparent" stroke="url(#gradient)" strokeWidth="2" />
   
        <circle cx="533.685" cy="533.685" r="420" fill="transparent" stroke="url(#gradient)" strokeWidth="6" />
     
        {smallCircles}
      </svg>
    </div>
  );
};

export default InteractiveComponent;






