import { useEffect, useRef, useState } from 'react'
import L from 'leaflet'
import 'leaflet/dist/leaflet.css'

const MapComponent = () => {
  const mapContainerRef = useRef<HTMLDivElement | null>(null)
  const mapRef = useRef<L.Map | null>(null)
  const [mapVisible, setMapVisible] = useState(true)

  // Definir un icono personalizado
  const customIcon = new L.Icon({
    iconUrl: '/Ubi.svg', // Ruta directa al archivo en la carpeta public
    iconSize: [38, 38],
    iconAnchor: [22, 38],
    popupAnchor: [0, -38],
  })
  
  

  const initializeMap = () => {
    if (mapContainerRef.current && !mapRef.current) {
      mapRef.current = L.map(mapContainerRef.current).setView(
        [9.940182415337974, -84.12664523067905],
        13
      )

      L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution:
          '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
      }).addTo(mapRef.current)

      // Usar el icono personalizado para el marcador
      L.marker([9.940182415337974, -84.12664523067905], { icon: customIcon })
        .addTo(mapRef.current)
        .bindPopup('ISA S.A.')
        .openPopup()
    }
  }

  const destroyMap = () => {
    if (mapRef.current) {
      mapRef.current.remove()
      mapRef.current = null
    }
  }

  useEffect(() => {
    if (mapVisible) {
      initializeMap()
    } else {
      destroyMap()
    }
  }, [mapVisible])

  useEffect(() => {
    const handleSidebarOpened = () => setMapVisible(false)
    const handleSidebarClosed = () => setMapVisible(true)

    window.addEventListener('sidebarOpened', handleSidebarOpened)
    window.addEventListener('sidebarClosed', handleSidebarClosed)

    return () => {
      window.removeEventListener('sidebarOpened', handleSidebarOpened)
      window.removeEventListener('sidebarClosed', handleSidebarClosed)
      destroyMap() // Asegura que el mapa se limpie cuando el componente se desmonte
    }
  }, [])

  return (
    <div
      ref={mapContainerRef}
      id='map'
      className={`h-[300px] my-[48px] md:my-[80px] w-full px-[16px] z-0 ${
        !mapVisible ? 'hidden' : ''
      }`}
    ></div>
  )
}

export default MapComponent
