import { useState } from 'react';
import axios from 'axios';
import { IMAGES_URL } from '@/constants';

interface FormSectionProps {
  image1: {
    src: string;
  };
}

const FormSection: React.FC<FormSectionProps> = ({ image1 }) => {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    message: '',
  });

  const [errors, setErrors] = useState({
    name: '',
    email: '',
    message: '',
  });

  // Estado para controlar la visibilidad de los modales
  const [isSuccessModalOpen, setIsSuccessModalOpen] = useState(false);
  const [isErrorModalOpen, setIsErrorModalOpen] = useState(false);

  const validateForm = () => {
    let formIsValid = true;
    let newErrors = { name: '', email: '', message: '' };

    // Validar el nombre
    if (!formData.name.trim()) {
      newErrors.name = 'El nombre es obligatorio.';
      formIsValid = false;
    }

    // Validar el correo electrónico
    const emailPattern = /^[a-zA-Z0-9._-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}$/;
    if (!formData.email.trim()) {
      newErrors.email = 'El correo es obligatorio.';
      formIsValid = false;
    } else if (!emailPattern.test(formData.email)) {
      newErrors.email = 'El formato del correo es inválido.';
      formIsValid = false;
    }

    // Validar el mensaje
    if (!formData.message.trim()) {
      newErrors.message = 'El mensaje es obligatorio.';
      formIsValid = false;
    }

    setErrors(newErrors);
    return formIsValid;
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData({
      ...formData,
      [name]: value,
    });
  };

  const handleSubmit = async () => {
    if (!validateForm()) {
      return;
    }

    try {
      const apiUrl = `${import.meta.env.PUBLIC_STRAPI_IMAGES}/api/contact/send`;

      await axios.post(apiUrl, {
        name: formData.name,
        email: formData.email,
        subject: 'Nuevo mensaje de contacto',
        message: formData.message,
      });

      // Mostrar el modal de éxito al enviar correctamente el correo
      setIsSuccessModalOpen(true);
    } catch (error) {
      console.error('Error al enviar el correo', error);

      // Mostrar el modal de error si algo falla
      setIsErrorModalOpen(true);
    }
  };

  const closeModal = () => {
    setIsSuccessModalOpen(false);
    setIsErrorModalOpen(false);
  };

  const hasErrors = !!errors.name || !!errors.email || !!errors.message;

  return (
    <div className="flex flex-col min-h-screen px-[16px] bg-black text-white mt-[150px] md:px-[180px] xl:px-[170px] 2xl:px-[180px] 2xl:mb-[40px]">
      <div className="flex items-center mb-6">
        <div className="w-[10px] h-[50px] bg-gradient-to-b from-azulGradiente to-verdeGradiente"></div>
        <h2 className="text-[32px] md:text-[50px] font-bold uppercase ml-4 text-center md:text-left">CONTACTANOS</h2>
      </div>

      {/* Imagen para móviles */}
      <div className="block md:hidden">
        <img
          src={IMAGES_URL + image1}
          alt="Imagen de edificios"
          className="w-full h-[200px] object-cover rounded-lg mb-6"
        />
      </div>

      <div className="flex flex-col md:flex-row justify-center items-center md:items-start space-y-6 md:space-y-0 md:space-x-12">
        {/* Formulario */}
        <div className="flex flex-col space-y-6 w-full md:w-[517px] xl:w-[800px] 2xl:w-[850px] 2xl:mr-[110px] xl:mr-[80px]">
          <div className="flex flex-col space-y-2">
            <label htmlFor="name" className="text-[20px] md:text-[28px]">NOMBRE</label>
            <input
              type="text"
              id="name"
              name="name"
              placeholder="Primer y segundo nombre"
              className="bg-[#1C1C1C] text-[20px] md:text-[28px] p-4 rounded-tr-[20px] rounded-bl-[20px] h-[60px] md:h-[80px] text-white focus:outline-none"
              value={formData.name}
              onChange={handleInputChange}
            />
            {errors.name && <p className="text-red-500">{errors.name}</p>}
          </div>

          <div className="flex flex-col space-y-2">
            <label htmlFor="email" className="text-[20px] md:text-[28px]">CORREO</label>
            <input
              type="email"
              id="email"
              name="email"
              placeholder="correo@example.com"
              className="bg-[#1C1C1C] p-4 text-[20px] md:text-[28px] rounded-tr-[20px] rounded-bl-[20px] h-[60px] md:h-[80px] text-white focus:outline-none"
              value={formData.email}
              onChange={handleInputChange}
            />
            {errors.email && <p className="text-red-500">{errors.email}</p>}
          </div>

          <div className="flex flex-col space-y-4">
            <label htmlFor="message" className="text-[20px] md:text-[28px]">MENSAJE</label>
            <textarea
              id="message"
              name="message"
              placeholder="Inserte mensaje"
              className="bg-[#1C1C1C] p-4 text-[20px] md:text-[28px] rounded-tr-[20px] rounded-bl-[20px] h-[150px] md:h-[320px] mb-[100px] text-white focus:outline-none"
              value={formData.message}
              onChange={handleInputChange}
            ></textarea>
            {errors.message && <p className="text-red-500">{errors.message}</p>}
          </div>
        </div>

        {/* Imagen para pantallas medianas y más grandes */}
        <div className={`hidden md:block md:w-full xl:w-full relative transition-all duration-300 ${hasErrors ? 'transform scale-105' : ''}`}>
          <img
            src={IMAGES_URL + image1}
            alt="Imagen de edificios"
            className="w-[923px] h-[710px] xl:w-[923px] xl:h-[678px] mt-[10px] object-cover rounded-lg"
          />
          <button
            onClick={handleSubmit}
            className="self-start flex items-center justify-center bg-white text-[#0F7D87] p-4 absolute top-[40px] md:top[1120px] 2xl:top-[115%] 2xl:left-[-45%] xl:top-[115%] font-semibold xl:left-[-58%] text-center rounded-full w-[200px] h-[50px] text-[24px] left-[50%] transform translate-x-[-50%] hover:bg-gray-300"
          >
            ENVIAR
          </button>
        </div>

        {/* Botón para móviles */}
        <button
          onClick={handleSubmit}
          className="md:hidden self-center bg-white text-[#0F7D87] p-4 rounded-full w-[60%] text-[20px] font-bold mt-6"
        >
          ENVIAR
        </button>
      </div>

      {/* Modal de éxito */}
      {isSuccessModalOpen && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-white bg-opacity-25">
          <div className="bg-black rounded-lg p-6 text-center shadow-xl max-w-md w-[384px] h-[284px] ">
            <img src="/Check_ISA.png" alt="Check Verde" className="mx-auto mb-4" style={{ width: '60px', height: '60px' }} />
            <h2 className="text-[20px] font-semibold text-white mb-2">MENSAJE ENVIADO</h2>
            <p className="text-white text-[18px]">ISA ha recibido tu mensaje, pronto te estaremos contactando</p>
            <button
              onClick={closeModal}
              className="mt-6 bg-[#0F7D87]  text-white py-2 px-4 rounded-full hover:bg-opacity-90 w-full text-[24px] font-bold "
            >
              VOLVER ATRÁS
            </button>
          </div>
        </div>
      )}

      {/* Modal de error */}
      {isErrorModalOpen && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-white bg-opacity-25">
          <div className="bg-black rounded-lg p-6 text-center shadow-xl max-w-md w-[384px] h-[284px] ">
            <img src="/Error_ISA.png" alt="Error" className="mx-auto mb-4" style={{ width: '60px', height: '60px' }} />
            <h2 className="text-[20px] font-semibold text-white mb-2">MENSAJE ENVIADO</h2>
            <p className="text-white text-[18px]">ISA ha recibido tu mensaje, pronto te estaremos contactando</p>
            <button
              onClick={closeModal}
              className="mt-6 bg-red-600 text-white py-2 px-4 rounded-full hover:bg-opacity-90 w-full text-[24px] font-bold "
            >
              VOLVER ATRÁS
            </button>
          </div>
        </div>
      )}
    </div>
  );
};

export default FormSection;
