import { z } from "zod";
import { getApiContext } from "../store.js";
import { hasContentType } from "../utils.js";
import {
  ActionError,
  ActionInputError,
  callSafely
} from "./shared.js";
export * from "./shared.js";
import { z as z2 } from "zod";
import { getApiContext as getApiContext2 } from "../store.js";
function defineAction({
  accept,
  input: inputSchema,
  handler
}) {
  const serverHandler = accept === "form" ? getFormServerHandler(handler, inputSchema) : getJsonServerHandler(handler, inputSchema);
  Object.assign(serverHandler, {
    safe: async (unparsedInput) => {
      return callSafely(() => serverHandler(unparsedInput));
    }
  });
  return serverHandler;
}
function getFormServerHandler(handler, inputSchema) {
  return async (unparsedInput) => {
    if (!(unparsedInput instanceof FormData)) {
      throw new ActionError({
        code: "UNSUPPORTED_MEDIA_TYPE",
        message: "This action only accepts FormData."
      });
    }
    if (!(inputSchema instanceof z.ZodObject)) return await handler(unparsedInput);
    const parsed = await inputSchema.safeParseAsync(formDataToObject(unparsedInput, inputSchema));
    if (!parsed.success) {
      throw new ActionInputError(parsed.error.issues);
    }
    return await handler(parsed.data);
  };
}
function getJsonServerHandler(handler, inputSchema) {
  return async (unparsedInput) => {
    const context = getApiContext();
    const contentType = context.request.headers.get("content-type");
    if (!contentType || !hasContentType(contentType, ["application/json"])) {
      throw new ActionError({
        code: "UNSUPPORTED_MEDIA_TYPE",
        message: "This action only accepts JSON."
      });
    }
    if (!inputSchema) return await handler(unparsedInput);
    const parsed = await inputSchema.safeParseAsync(unparsedInput);
    if (!parsed.success) {
      throw new ActionInputError(parsed.error.issues);
    }
    return await handler(parsed.data);
  };
}
function formDataToObject(formData, schema) {
  const obj = {};
  for (const [key, baseValidator] of Object.entries(schema.shape)) {
    let validator = baseValidator;
    if (baseValidator instanceof z.ZodOptional || baseValidator instanceof z.ZodNullable) {
      validator = baseValidator._def.innerType;
    }
    if (validator instanceof z.ZodBoolean) {
      obj[key] = formData.has(key);
    } else if (validator instanceof z.ZodArray) {
      obj[key] = handleFormDataGetAll(key, formData, validator);
    } else {
      obj[key] = handleFormDataGet(key, formData, validator, baseValidator);
    }
  }
  return obj;
}
function handleFormDataGetAll(key, formData, validator) {
  const entries = Array.from(formData.getAll(key));
  const elementValidator = validator._def.type;
  if (elementValidator instanceof z.ZodNumber) {
    return entries.map(Number);
  } else if (elementValidator instanceof z.ZodBoolean) {
    return entries.map(Boolean);
  }
  return entries;
}
function handleFormDataGet(key, formData, validator, baseValidator) {
  const value = formData.get(key);
  if (!value) {
    return baseValidator instanceof z.ZodOptional ? void 0 : null;
  }
  return validator instanceof z.ZodNumber ? Number(value) : value;
}
export {
  defineAction,
  formDataToObject,
  getApiContext2 as getApiContext,
  z2 as z
};
