import { yellow } from "kleur/colors";
import { defineMiddleware } from "../../core/middleware/index.js";
import { ApiContextStorage } from "./store.js";
import { formContentTypes, getAction, hasContentType } from "./utils.js";
import { callSafely } from "./virtual/shared.js";
const onRequest = defineMiddleware(async (context, next) => {
  const locals = context.locals;
  if (context.request.method === "GET") {
    return nextWithLocalsStub(next, locals);
  }
  if (context.request.method === "POST" && context.request.body === null) {
    return nextWithStaticStub(next, locals);
  }
  if (locals._actionsInternal) return next();
  const { request, url } = context;
  const contentType = request.headers.get("Content-Type");
  if (url.pathname.startsWith("/_actions")) return nextWithLocalsStub(next, locals);
  if (!contentType || !hasContentType(contentType, formContentTypes)) {
    return nextWithLocalsStub(next, locals);
  }
  const formData = await request.clone().formData();
  const actionPath = formData.get("_astroAction");
  if (typeof actionPath !== "string") return nextWithLocalsStub(next, locals);
  const actionPathKeys = actionPath.replace("/_actions/", "").split(".");
  const action = await getAction(actionPathKeys);
  if (!action) return nextWithLocalsStub(next, locals);
  const result = await ApiContextStorage.run(context, () => callSafely(() => action(formData)));
  const actionsInternal = {
    getActionResult: (actionFn) => {
      if (actionFn.toString() !== actionPath) return Promise.resolve(void 0);
      return result;
    }
  };
  Object.defineProperty(locals, "_actionsInternal", { writable: false, value: actionsInternal });
  const response = await next();
  if (result.error) {
    return new Response(response.body, {
      status: result.error.status,
      statusText: result.error.name,
      headers: response.headers
    });
  }
  return response;
});
function nextWithStaticStub(next, locals) {
  Object.defineProperty(locals, "_actionsInternal", {
    writable: false,
    value: {
      getActionResult: () => {
        console.warn(
          yellow("[astro:actions]"),
          "`getActionResult()` should not be called on prerendered pages. Astro can only handle actions for pages rendered on-demand."
        );
        return void 0;
      }
    }
  });
  return next();
}
function nextWithLocalsStub(next, locals) {
  Object.defineProperty(locals, "_actionsInternal", {
    writable: false,
    value: {
      getActionResult: () => void 0
    }
  });
  return next();
}
export {
  onRequest
};
