import { L as LoadWasmOptions } from './chunk-index.mjs';

// ## Interfaces

/**
 * Info associated with nodes by the ecosystem.
 *
 * This space is guaranteed to never be specified by unist or specifications
 * implementing unist.
 * But you can use it in utilities and plugins to store data.
 *
 * This type can be augmented to register custom data.
 * For example:
 *
 * ```ts
 * declare module 'unist' {
 *   interface Data {
 *     // `someNode.data.myId` is typed as `number | undefined`
 *     myId?: number | undefined
 *   }
 * }
 * ```
 */
interface Data$1 {}

/**
 * One place in a source file.
 */
interface Point {
    /**
     * Line in a source file (1-indexed integer).
     */
    line: number;

    /**
     * Column in a source file (1-indexed integer).
     */
    column: number;
    /**
     * Character in a source file (0-indexed integer).
     */
    offset?: number | undefined;
}

/**
 * Position of a node in a source document.
 *
 * A position is a range between two points.
 */
interface Position$1 {
    /**
     * Place of the first character of the parsed source region.
     */
    start: Point;

    /**
     * Place of the first character after the parsed source region.
     */
    end: Point;
}

/**
 * Abstract unist node.
 *
 * The syntactic unit in unist syntax trees are called nodes.
 *
 * This interface is supposed to be extended.
 * If you can use {@link Literal} or {@link Parent}, you should.
 * But for example in markdown, a `thematicBreak` (`***`), is neither literal
 * nor parent, but still a node.
 */
interface Node$1 {
    /**
     * Node type.
     */
    type: string;

    /**
     * Info from the ecosystem.
     */
    data?: Data$1 | undefined;

    /**
     * Position of a node in a source document.
     *
     * Nodes that are generated (not in the original source document) must not
     * have a position.
     */
    position?: Position$1 | undefined;
}

// ## Interfaces

/**
 * Info associated with hast nodes by the ecosystem.
 *
 * This space is guaranteed to never be specified by unist or hast.
 * But you can use it in utilities and plugins to store data.
 *
 * This type can be augmented to register custom data.
 * For example:
 *
 * ```ts
 * declare module 'hast' {
 *   interface Data {
 *     // `someNode.data.myId` is typed as `number | undefined`
 *     myId?: number | undefined
 *   }
 * }
 * ```
 */
interface Data extends Data$1 {}

/**
 * Info associated with an element.
 */
interface Properties {
    [PropertyName: string]: boolean | number | string | null | undefined | Array<string | number>;
}

// ## Content maps

/**
 * Union of registered hast nodes that can occur in {@link Element}.
 *
 * To register mote custom hast nodes, add them to {@link ElementContentMap}.
 * They will be automatically added here.
 */
type ElementContent = ElementContentMap[keyof ElementContentMap];

/**
 * Registry of all hast nodes that can occur as children of {@link Element}.
 *
 * For a union of all {@link Element} children, see {@link ElementContent}.
 */
interface ElementContentMap {
    comment: Comment;
    element: Element;
    text: Text;
}

/**
 * Union of registered hast nodes that can occur in {@link Root}.
 *
 * To register custom hast nodes, add them to {@link RootContentMap}.
 * They will be automatically added here.
 */
type RootContent = RootContentMap[keyof RootContentMap];

/**
 * Registry of all hast nodes that can occur as children of {@link Root}.
 *
 * > 👉 **Note**: {@link Root} does not need to be an entire document.
 * > it can also be a fragment.
 *
 * For a union of all {@link Root} children, see {@link RootContent}.
 */
interface RootContentMap {
    comment: Comment;
    doctype: Doctype;
    element: Element;
    text: Text;
}

/**
 * Union of registered hast nodes.
 *
 * To register custom hast nodes, add them to {@link RootContentMap} and other
 * places where relevant.
 * They will be automatically added here.
 */
type Nodes = Root | RootContent;

// ## Abstract nodes

/**
 * Abstract hast node.
 *
 * This interface is supposed to be extended.
 * If you can use {@link Literal} or {@link Parent}, you should.
 * But for example in HTML, a `Doctype` is neither literal nor parent, but
 * still a node.
 *
 * To register custom hast nodes, add them to {@link RootContentMap} and other
 * places where relevant (such as {@link ElementContentMap}).
 *
 * For a union of all registered hast nodes, see {@link Nodes}.
 */
interface Node extends Node$1 {
    /**
     * Info from the ecosystem.
     */
    data?: Data | undefined;
}

/**
 * Abstract hast node that contains the smallest possible value.
 *
 * This interface is supposed to be extended if you make custom hast nodes.
 *
 * For a union of all registered hast literals, see {@link Literals}.
 */
interface Literal extends Node {
    /**
     * Plain-text value.
     */
    value: string;
}

/**
 * Abstract hast node that contains other hast nodes (*children*).
 *
 * This interface is supposed to be extended if you make custom hast nodes.
 *
 * For a union of all registered hast parents, see {@link Parents}.
 */
interface Parent extends Node {
    /**
     * List of children.
     */
    children: RootContent[];
}

// ## Concrete nodes

/**
 * HTML comment.
 */
interface Comment extends Literal {
    /**
     * Node type of HTML comments in hast.
     */
    type: "comment";
    /**
     * Data associated with the comment.
     */
    data?: CommentData | undefined;
}

/**
 * Info associated with hast comments by the ecosystem.
 */
interface CommentData extends Data {}

/**
 * HTML document type.
 */
interface Doctype extends Node$1 {
    /**
     * Node type of HTML document types in hast.
     */
    type: "doctype";
    /**
     * Data associated with the doctype.
     */
    data?: DoctypeData | undefined;
}

/**
 * Info associated with hast doctypes by the ecosystem.
 */
interface DoctypeData extends Data {}

/**
 * HTML element.
 */
interface Element extends Parent {
    /**
     * Node type of elements.
     */
    type: "element";
    /**
     * Tag name (such as `'body'`) of the element.
     */
    tagName: string;
    /**
     * Info associated with the element.
     */
    properties: Properties;
    /**
     * Children of element.
     */
    children: ElementContent[];
    /**
     * When the `tagName` field is `'template'`, a `content` field can be
     * present.
     */
    content?: Root | undefined;
    /**
     * Data associated with the element.
     */
    data?: ElementData | undefined;
}

/**
 * Info associated with hast elements by the ecosystem.
 */
interface ElementData extends Data {}

/**
 * Document fragment or a whole document.
 *
 * Should be used as the root of a tree and must not be used as a child.
 *
 * Can also be used as the value for the content field on a `'template'` element.
 */
interface Root extends Parent {
    /**
     * Node type of hast root.
     */
    type: "root";
    /**
     * Children of root.
     */
    children: RootContent[];
    /**
     * Data associated with the hast root.
     */
    data?: RootData | undefined;
}

/**
 * Info associated with hast root nodes by the ecosystem.
 */
interface RootData extends Data {}

/**
 * HTML character data (plain text).
 */
interface Text extends Literal {
    /**
     * Node type of HTML character data (plain text) in hast.
     */
    type: "text";
    /**
     * Data associated with the text.
     */
    data?: TextData | undefined;
}

/**
 * Info associated with hast texts by the ecosystem.
 */
interface TextData extends Data {}

/**
 * A union of given const enum values.
*/
type OrMask<T extends number> = number;

interface IOnigLib {
    createOnigScanner(sources: string[]): OnigScanner;
    createOnigString(str: string): OnigString;
}
interface IOnigCaptureIndex {
    start: number;
    end: number;
    length: number;
}
interface IOnigMatch {
    index: number;
    captureIndices: IOnigCaptureIndex[];
}
declare const enum FindOption {
    None = 0,
    /**
     * equivalent of ONIG_OPTION_NOT_BEGIN_STRING: (str) isn't considered as begin of string (* fail \A)
     */
    NotBeginString = 1,
    /**
     * equivalent of ONIG_OPTION_NOT_END_STRING: (end) isn't considered as end of string (* fail \z, \Z)
     */
    NotEndString = 2,
    /**
     * equivalent of ONIG_OPTION_NOT_BEGIN_POSITION: (start) isn't considered as start position of search (* fail \G)
     */
    NotBeginPosition = 4,
    /**
     * used for debugging purposes.
     */
    DebugCall = 8
}
interface OnigScanner {
    findNextMatchSync(string: string | OnigString, startPosition: number, options: OrMask<FindOption>): IOnigMatch | null;
    dispose?(): void;
}
interface OnigString {
    readonly content: string;
    dispose?(): void;
}

declare const ruleIdSymbol: unique symbol;
type RuleId = {
    __brand: typeof ruleIdSymbol;
};

declare class Theme {
    private readonly _colorMap;
    private readonly _defaults;
    private readonly _root;
    static createFromRawTheme(source: IRawTheme | undefined, colorMap?: string[]): Theme;
    static createFromParsedTheme(source: ParsedThemeRule[], colorMap?: string[]): Theme;
    private readonly _cachedMatchRoot;
    constructor(_colorMap: ColorMap, _defaults: StyleAttributes, _root: ThemeTrieElement);
    getColorMap(): string[];
    getDefaults(): StyleAttributes;
    match(scopePath: ScopeStack | null): StyleAttributes | null;
}
/**
 * Identifiers with a binary dot operator.
 * Examples: `baz` or `foo.bar`
*/
type ScopeName = string;
/**
 * An expression language of ScopePathStr with a binary comma (to indicate alternatives) operator.
 * Examples: `foo.bar boo.baz,quick quack`
*/
type ScopePattern = string;
/**
 * A TextMate theme.
 */
interface IRawTheme {
    readonly name?: string;
    readonly settings: IRawThemeSetting[];
}
/**
 * A single theme setting.
 */
interface IRawThemeSetting {
    readonly name?: string;
    readonly scope?: ScopePattern | ScopePattern[];
    readonly settings: {
        readonly fontStyle?: string;
        readonly foreground?: string;
        readonly background?: string;
    };
}
declare class ScopeStack {
    readonly parent: ScopeStack | null;
    readonly scopeName: ScopeName;
    static push(path: ScopeStack | null, scopeNames: ScopeName[]): ScopeStack | null;
    static from(first: ScopeName, ...segments: ScopeName[]): ScopeStack;
    static from(...segments: ScopeName[]): ScopeStack | null;
    constructor(parent: ScopeStack | null, scopeName: ScopeName);
    push(scopeName: ScopeName): ScopeStack;
    getSegments(): ScopeName[];
    toString(): string;
    extends(other: ScopeStack): boolean;
    getExtensionIfDefined(base: ScopeStack | null): string[] | undefined;
}
declare class StyleAttributes {
    readonly fontStyle: OrMask<FontStyle$1>;
    readonly foregroundId: number;
    readonly backgroundId: number;
    constructor(fontStyle: OrMask<FontStyle$1>, foregroundId: number, backgroundId: number);
}
declare class ParsedThemeRule {
    readonly scope: ScopeName;
    readonly parentScopes: ScopeName[] | null;
    readonly index: number;
    readonly fontStyle: OrMask<FontStyle$1>;
    readonly foreground: string | null;
    readonly background: string | null;
    constructor(scope: ScopeName, parentScopes: ScopeName[] | null, index: number, fontStyle: OrMask<FontStyle$1>, foreground: string | null, background: string | null);
}
declare const enum FontStyle$1 {
    NotSet = -1,
    None = 0,
    Italic = 1,
    Bold = 2,
    Underline = 4,
    Strikethrough = 8
}
declare class ColorMap {
    private readonly _isFrozen;
    private _lastColorId;
    private _id2color;
    private _color2id;
    constructor(_colorMap?: string[]);
    getId(color: string | null): number;
    getColorMap(): string[];
}
declare class ThemeTrieElementRule {
    scopeDepth: number;
    parentScopes: ScopeName[] | null;
    fontStyle: number;
    foreground: number;
    background: number;
    constructor(scopeDepth: number, parentScopes: ScopeName[] | null, fontStyle: number, foreground: number, background: number);
    clone(): ThemeTrieElementRule;
    static cloneArr(arr: ThemeTrieElementRule[]): ThemeTrieElementRule[];
    acceptOverwrite(scopeDepth: number, fontStyle: number, foreground: number, background: number): void;
}
interface ITrieChildrenMap {
    [segment: string]: ThemeTrieElement;
}
declare class ThemeTrieElement {
    private readonly _mainRule;
    private readonly _children;
    private readonly _rulesWithParentScopes;
    constructor(_mainRule: ThemeTrieElementRule, rulesWithParentScopes?: ThemeTrieElementRule[], _children?: ITrieChildrenMap);
    private static _sortBySpecificity;
    private static _cmpBySpecificity;
    match(scope: ScopeName): ThemeTrieElementRule[];
    insert(scopeDepth: number, scope: ScopeName, parentScopes: ScopeName[] | null, fontStyle: number, foreground: number, background: number): void;
    private _doInsertHere;
}

interface IRawGrammar extends ILocatable {
    repository: IRawRepository;
    readonly scopeName: ScopeName;
    readonly patterns: IRawRule[];
    readonly injections?: {
        [expression: string]: IRawRule;
    };
    readonly injectionSelector?: string;
    readonly fileTypes?: string[];
    readonly name?: string;
    readonly firstLineMatch?: string;
}
/**
 * Allowed values:
 * * Scope Name, e.g. `source.ts`
 * * Top level scope reference, e.g. `source.ts#entity.name.class`
 * * Relative scope reference, e.g. `#entity.name.class`
 * * self, e.g. `$self`
 * * base, e.g. `$base`
 */
type IncludeString = string;
type RegExpString = string;
interface IRawRepositoryMap {
    [name: string]: IRawRule;
    $self: IRawRule;
    $base: IRawRule;
}
type IRawRepository = IRawRepositoryMap & ILocatable;
interface IRawRule extends ILocatable {
    id?: RuleId;
    readonly include?: IncludeString;
    readonly name?: ScopeName;
    readonly contentName?: ScopeName;
    readonly match?: RegExpString;
    readonly captures?: IRawCaptures;
    readonly begin?: RegExpString;
    readonly beginCaptures?: IRawCaptures;
    readonly end?: RegExpString;
    readonly endCaptures?: IRawCaptures;
    readonly while?: RegExpString;
    readonly whileCaptures?: IRawCaptures;
    readonly patterns?: IRawRule[];
    readonly repository?: IRawRepository;
    readonly applyEndPatternLast?: boolean;
}
type IRawCaptures = IRawCapturesMap & ILocatable;
interface IRawCapturesMap {
    [captureId: string]: IRawRule;
}
interface ILocation {
    readonly filename: string;
    readonly line: number;
    readonly char: number;
}
interface ILocatable {
    readonly $vscodeTextmateLocation?: ILocation;
}

declare const enum StandardTokenType {
    Other = 0,
    Comment = 1,
    String = 2,
    RegEx = 3
}

/**
 * A registry helper that can locate grammar file paths given scope names.
 */
interface RegistryOptions {
    onigLib: Promise<IOnigLib>;
    theme?: IRawTheme;
    colorMap?: string[];
    loadGrammar(scopeName: ScopeName): Promise<IRawGrammar | undefined | null>;
    getInjections?(scopeName: ScopeName): ScopeName[] | undefined;
}
/**
 * A map from scope name to a language id. Please do not use language id 0.
 */
interface IEmbeddedLanguagesMap {
    [scopeName: string]: number;
}
/**
 * A map from selectors to token types.
 */
interface ITokenTypeMap {
    [selector: string]: StandardTokenType;
}
interface IGrammarConfiguration {
    embeddedLanguages?: IEmbeddedLanguagesMap;
    tokenTypes?: ITokenTypeMap;
    balancedBracketSelectors?: string[];
    unbalancedBracketSelectors?: string[];
}
/**
 * The registry that will hold all grammars.
 */
declare class Registry {
    private readonly _options;
    private readonly _syncRegistry;
    private readonly _ensureGrammarCache;
    constructor(options: RegistryOptions);
    dispose(): void;
    /**
     * Change the theme. Once called, no previous `ruleStack` should be used anymore.
     */
    setTheme(theme: IRawTheme, colorMap?: string[]): void;
    /**
     * Returns a lookup array for color ids.
     */
    getColorMap(): string[];
    /**
     * Load the grammar for `scopeName` and all referenced included grammars asynchronously.
     * Please do not use language id 0.
     */
    loadGrammarWithEmbeddedLanguages(initialScopeName: ScopeName, initialLanguage: number, embeddedLanguages: IEmbeddedLanguagesMap): Promise<IGrammar | null>;
    /**
     * Load the grammar for `scopeName` and all referenced included grammars asynchronously.
     * Please do not use language id 0.
     */
    loadGrammarWithConfiguration(initialScopeName: ScopeName, initialLanguage: number, configuration: IGrammarConfiguration): Promise<IGrammar | null>;
    /**
     * Load the grammar for `scopeName` and all referenced included grammars asynchronously.
     */
    loadGrammar(initialScopeName: ScopeName): Promise<IGrammar | null>;
    private _loadGrammar;
    private _loadSingleGrammar;
    private _doLoadSingleGrammar;
    /**
     * Adds a rawGrammar.
     */
    addGrammar(rawGrammar: IRawGrammar, injections?: string[], initialLanguage?: number, embeddedLanguages?: IEmbeddedLanguagesMap | null): Promise<IGrammar>;
    /**
     * Get the grammar for `scopeName`. The grammar must first be created via `loadGrammar` or `addGrammar`.
     */
    private _grammarForScopeName;
}
/**
 * A grammar
 */
interface IGrammar {
    /**
     * Tokenize `lineText` using previous line state `prevState`.
     */
    tokenizeLine(lineText: string, prevState: StateStack | null, timeLimit?: number): ITokenizeLineResult;
    /**
     * Tokenize `lineText` using previous line state `prevState`.
     * The result contains the tokens in binary format, resolved with the following information:
     *  - language
     *  - token type (regex, string, comment, other)
     *  - font style
     *  - foreground color
     *  - background color
     * e.g. for getting the languageId: `(metadata & MetadataConsts.LANGUAGEID_MASK) >>> MetadataConsts.LANGUAGEID_OFFSET`
     */
    tokenizeLine2(lineText: string, prevState: StateStack | null, timeLimit?: number): ITokenizeLineResult2;
}
interface ITokenizeLineResult {
    readonly tokens: IToken[];
    /**
     * The `prevState` to be passed on to the next line tokenization.
     */
    readonly ruleStack: StateStack;
    /**
     * Did tokenization stop early due to reaching the time limit.
     */
    readonly stoppedEarly: boolean;
}
interface ITokenizeLineResult2 {
    /**
     * The tokens in binary format. Each token occupies two array indices. For token i:
     *  - at offset 2*i => startIndex
     *  - at offset 2*i + 1 => metadata
     *
     */
    readonly tokens: Uint32Array;
    /**
     * The `prevState` to be passed on to the next line tokenization.
     */
    readonly ruleStack: StateStack;
    /**
     * Did tokenization stop early due to reaching the time limit.
     */
    readonly stoppedEarly: boolean;
}
interface IToken {
    startIndex: number;
    readonly endIndex: number;
    readonly scopes: string[];
}
/**
 * **IMPORTANT** - Immutable!
 */
interface StateStack {
    _stackElementBrand: void;
    readonly depth: number;
    clone(): StateStack;
    equals(other: StateStack): boolean;
}
declare const INITIAL: StateStack;

type Awaitable<T> = T | Promise<T>;
type MaybeGetter<T> = Awaitable<MaybeModule<T>> | (() => Awaitable<MaybeModule<T>>);
type MaybeModule<T> = T | {
    default: T;
};
type MaybeArray<T> = T | T[];
type RequireKeys<T, K extends keyof T> = Omit<T, K> & Required<Pick<T, K>>;
interface Nothing {
}
/**
 * type StringLiteralUnion<'foo'> = 'foo' | string
 * This has auto completion whereas `'foo' | string` doesn't
 * Adapted from https://github.com/microsoft/TypeScript/issues/29729
 */
type StringLiteralUnion<T extends U, U = string> = T | (U & Nothing);

type PlainTextLanguage = 'text' | 'plaintext' | 'txt';
type AnsiLanguage = 'ansi';
type SpecialLanguage = PlainTextLanguage | AnsiLanguage;
type LanguageInput = MaybeGetter<MaybeArray<LanguageRegistration>>;
type ResolveBundleKey<T extends string> = [T] extends [never] ? string : T;
interface LanguageRegistration extends IRawGrammar {
    name: string;
    scopeName: string;
    displayName?: string;
    aliases?: string[];
    /**
     * A list of languages the current language embeds.
     * If manually specifying languages to load, make sure to load the embedded
     * languages for each parent language.
     */
    embeddedLangs?: string[];
    /**
     * A list of languages that embed the current language.
     * Unlike `embeddedLangs`, the embedded languages will not be loaded automatically.
     */
    embeddedLangsLazy?: string[];
    balancedBracketSelectors?: string[];
    unbalancedBracketSelectors?: string[];
    foldingStopMarker?: string;
    foldingStartMarker?: string;
    /**
     * Inject this language to other scopes.
     * Same as `injectTo` in VSCode's `contributes.grammars`.
     *
     * @see https://code.visualstudio.com/api/language-extensions/syntax-highlight-guide#injection-grammars
     */
    injectTo?: string[];
}
interface BundledLanguageInfo {
    id: string;
    name: string;
    import: DynamicImportLanguageRegistration;
    aliases?: string[];
}
type DynamicImportLanguageRegistration = () => Promise<{
    default: LanguageRegistration[];
}>;

type SpecialTheme = 'none';
type ThemeInput = MaybeGetter<ThemeRegistrationAny>;
interface ThemeRegistrationRaw extends IRawTheme, Partial<Omit<ThemeRegistration, 'name' | 'settings'>> {
}
interface ThemeRegistration extends Partial<ThemeRegistrationResolved> {
}
interface ThemeRegistrationResolved extends IRawTheme {
    /**
     * Theme name
     */
    name: string;
    /**
     * Display name
     *
     * @field shiki custom property
     */
    displayName?: string;
    /**
     * Light/dark theme
     *
     * @field shiki custom property
     */
    type: 'light' | 'dark';
    /**
     * Token rules
     */
    settings: IRawThemeSetting[];
    /**
     * Same as `settings`, will use as fallback if `settings` is not present.
     */
    tokenColors?: IRawThemeSetting[];
    /**
     * Default foreground color
     *
     * @field shiki custom property
     */
    fg: string;
    /**
     * Background color
     *
     * @field shiki custom property
     */
    bg: string;
    /**
     * A map of color names to new color values.
     *
     * The color key starts with '#' and should be lowercased.
     *
     * @field shiki custom property
     */
    colorReplacements?: Record<string, string>;
    /**
     * Color map of VS Code options
     *
     * Will be used by shiki on `lang: 'ansi'` to find ANSI colors, and to find the default foreground/background colors.
     */
    colors?: Record<string, string>;
    /**
     * JSON schema path
     *
     * @field not used by shiki
     */
    $schema?: string;
    /**
     * Enable semantic highlighting
     *
     * @field not used by shiki
     */
    semanticHighlighting?: boolean;
    /**
     * Tokens for semantic highlighting
     *
     * @field not used by shiki
     */
    semanticTokenColors?: Record<string, string>;
}
type ThemeRegistrationAny = ThemeRegistrationRaw | ThemeRegistration | ThemeRegistrationResolved;
type DynamicImportThemeRegistration = () => Promise<{
    default: ThemeRegistration;
}>;
interface BundledThemeInfo {
    id: string;
    displayName: string;
    type: 'light' | 'dark';
    import: DynamicImportThemeRegistration;
}

interface TransformerOptions {
    /**
     * Transformers for the Shiki pipeline.
     */
    transformers?: ShikiTransformer[];
}
interface ShikiTransformerContextMeta {
}
/**
 * Common transformer context for all transformers hooks
 */
interface ShikiTransformerContextCommon {
    meta: ShikiTransformerContextMeta;
    options: CodeToHastOptions;
    codeToHast: (code: string, options: CodeToHastOptions) => Root;
    codeToTokens: (code: string, options: CodeToTokensOptions) => TokensResult;
}
interface ShikiTransformerContextSource extends ShikiTransformerContextCommon {
    readonly source: string;
}
/**
 * Transformer context for HAST related hooks
 */
interface ShikiTransformerContext extends ShikiTransformerContextSource {
    readonly tokens: ThemedToken[][];
    readonly root: Root;
    readonly pre: Element;
    readonly code: Element;
    readonly lines: Element[];
    readonly structure: CodeToHastOptions['structure'];
    /**
     * Utility to append class to a hast node
     *
     * If the `property.class` is a string, it will be splitted by space and converted to an array.
     */
    addClassToHast: (hast: Element, className: string | string[]) => Element;
}
interface ShikiTransformer {
    /**
     * Name of the transformer
     */
    name?: string;
    /**
     * Transform the raw input code before passing to the highlighter.
     */
    preprocess?: (this: ShikiTransformerContextCommon, code: string, options: CodeToHastOptions) => string | void;
    /**
     * Transform the full tokens list before converting to HAST.
     * Return a new tokens list will replace the original one.
     */
    tokens?: (this: ShikiTransformerContextSource, tokens: ThemedToken[][]) => ThemedToken[][] | void;
    /**
     * Transform the entire generated HAST tree. Return a new Node will replace the original one.
     */
    root?: (this: ShikiTransformerContext, hast: Root) => Root | void;
    /**
     * Transform the `<pre>` element. Return a new Node will replace the original one.
     */
    pre?: (this: ShikiTransformerContext, hast: Element) => Element | void;
    /**
     * Transform the `<code>` element. Return a new Node will replace the original one.
     */
    code?: (this: ShikiTransformerContext, hast: Element) => Element | void;
    /**
     * Transform each line `<span class="line">` element.
     *
     * @param hast
     * @param line 1-based line number
     */
    line?: (this: ShikiTransformerContext, hast: Element, line: number) => Element | void;
    /**
     * Transform each token `<span>` element.
     */
    span?: (this: ShikiTransformerContext, hast: Element, line: number, col: number, lineElement: Element) => Element | void;
    /**
     * Transform the generated HTML string before returning.
     * This hook will only be called with `codeToHtml`.
     */
    postprocess?: (this: ShikiTransformerContextCommon, html: string, options: CodeToHastOptions) => string | void;
}

interface DecorationOptions {
    /**
     * Custom decorations to wrap highlighted tokens with.
     */
    decorations?: DecorationItem[];
}
interface DecorationItem {
    /**
     * Start offset or position of the decoration.
     */
    start: OffsetOrPosition;
    /**
     * End offset or position of the decoration.
     *
     * If the
     */
    end: OffsetOrPosition;
    /**
     * Tag name of the element to create.
     * @default 'span'
     */
    tagName?: string;
    /**
     * Properties of the element to create.
     */
    properties?: Element['properties'];
    /**
     * A custom function to transform the element after it has been created.
     */
    transform?: (element: Element, type: DecorationTransformType) => Element | void;
    /**
     * By default when the decoration contains only one token, the decoration will be applied to the token.
     *
     * Set to `true` to always wrap the token with a new element
     *
     * @default false
     */
    alwaysWrap?: boolean;
}
interface ResolvedDecorationItem extends Omit<DecorationItem, 'start' | 'end'> {
    start: ResolvedPosition;
    end: ResolvedPosition;
}
type DecorationTransformType = 'wrapper' | 'line' | 'token';
interface Position {
    line: number;
    character: number;
}
type Offset = number;
type OffsetOrPosition = Position | Offset;
interface ResolvedPosition extends Position {
    offset: Offset;
}

interface HighlighterCoreOptions {
    /**
     * Theme names, or theme registration objects to be loaded upfront.
     */
    themes?: ThemeInput[];
    /**
     * Language names, or language registration objects to be loaded upfront.
     */
    langs?: LanguageInput[];
    /**
     * Alias of languages
     * @example { 'my-lang': 'javascript' }
     */
    langAlias?: Record<string, string>;
    /**
     * Load wasm file from a custom path or using a custom function.
     */
    loadWasm?: LoadWasmOptions;
}
interface BundledHighlighterOptions<L extends string, T extends string> {
    /**
     * Theme registation
     *
     * @default []
     */
    themes: (ThemeInput | StringLiteralUnion<T>)[];
    /**
     * Language registation
     *
     * @default []
     */
    langs: (LanguageInput | StringLiteralUnion<L> | SpecialLanguage)[];
    /**
     * Alias of languages
     * @example { 'my-lang': 'javascript' }
     */
    langAlias?: Record<string, StringLiteralUnion<L>>;
}
interface CodeOptionsSingleTheme<Themes extends string = string> {
    theme: ThemeRegistrationAny | StringLiteralUnion<Themes>;
}
interface CodeOptionsMultipleThemes<Themes extends string = string> {
    /**
     * A map of color names to themes.
     * This allows you to specify multiple themes for the generated code.
     *
     * ```ts
     * highlighter.codeToHtml(code, {
     *   lang: 'js',
     *   themes: {
     *     light: 'vitesse-light',
     *     dark: 'vitesse-dark',
     *   }
     * })
     * ```
     *
     * Will generate:
     *
     * ```html
     * <span style="color:#111;--shiki-dark:#fff;">code</span>
     * ```
     *
     * @see https://github.com/shikijs/shiki#lightdark-dual-themes
     */
    themes: Partial<Record<string, ThemeRegistrationAny | StringLiteralUnion<Themes>>>;
    /**
     * The default theme applied to the code (via inline `color` style).
     * The rest of the themes are applied via CSS variables, and toggled by CSS overrides.
     *
     * For example, if `defaultColor` is `light`, then `light` theme is applied to the code,
     * and the `dark` theme and other custom themes are applied via CSS variables:
     *
     * ```html
     * <span style="color:#{light};--shiki-dark:#{dark};--shiki-custom:#{custom};">code</span>
     * ```
     *
     * When set to `false`, no default styles will be applied, and totally up to users to apply the styles:
     *
     * ```html
     * <span style="--shiki-light:#{light};--shiki-dark:#{dark};--shiki-custom:#{custom};">code</span>
     * ```
     *
     *
     * @default 'light'
     */
    defaultColor?: StringLiteralUnion<'light' | 'dark'> | false;
    /**
     * Prefix of CSS variables used to store the color of the other theme.
     *
     * @default '--shiki-'
     */
    cssVariablePrefix?: string;
}
type CodeOptionsThemes<Themes extends string = string> = CodeOptionsSingleTheme<Themes> | CodeOptionsMultipleThemes<Themes>;
type CodeToHastOptions<Languages extends string = string, Themes extends string = string> = CodeToHastOptionsCommon<Languages> & CodeOptionsThemes<Themes> & CodeOptionsMeta;
interface CodeToHastOptionsCommon<Languages extends string = string> extends TransformerOptions, DecorationOptions, Pick<TokenizeWithThemeOptions, 'colorReplacements' | 'tokenizeMaxLineLength' | 'tokenizeTimeLimit'> {
    lang: StringLiteralUnion<Languages | SpecialLanguage>;
    /**
     * Merge whitespace tokens to saving extra `<span>`.
     *
     * When set to true, it will merge whitespace tokens with the next token.
     * When set to false, it keep the output as-is.
     * When set to `never`, it will force to separate leading and trailing spaces from tokens.
     *
     * @default true
     */
    mergeWhitespaces?: boolean | 'never';
    /**
     * The structure of the generated HAST and HTML.
     *
     * - `classic`: The classic structure with `<pre>` and `<code>` elements, each line wrapped with a `<span class="line">` element.
     * - `inline`: All tokens are rendered as `<span>`, line breaks are rendered as `<br>`. No `<pre>` or `<code>` elements. Default forground and background colors are not applied.
     *
     * @default 'classic'
     */
    structure?: 'classic' | 'inline';
}
interface CodeOptionsMeta {
    /**
     * Meta data passed to Shiki, usually used by plugin integrations to pass the code block header.
     *
     * Key values in meta will be serialized to the attributes of the root `<pre>` element.
     *
     * Keys starting with `_` will be ignored.
     *
     * A special key `__raw` key will be used to pass the raw code block header (if the integration supports it).
     */
    meta?: {
        /**
         * Raw string of the code block header.
         */
        __raw?: string;
        [key: string]: any;
    };
}
interface CodeToHastRenderOptionsCommon extends TransformerOptions, Omit<TokensResult, 'tokens'> {
    lang?: string;
    langId?: string;
}
type CodeToHastRenderOptions = CodeToHastRenderOptionsCommon & CodeToHastOptions;

interface CodeToTokensBaseOptions<Languages extends string = string, Themes extends string = string> extends TokenizeWithThemeOptions {
    lang?: Languages | SpecialLanguage;
    theme?: Themes | ThemeRegistrationAny | SpecialTheme;
}
type CodeToTokensOptions<Languages extends string = string, Themes extends string = string> = Omit<CodeToTokensBaseOptions<Languages, Themes>, 'theme'> & CodeOptionsThemes<Themes>;
interface CodeToTokensWithThemesOptions<Languages = string, Themes = string> {
    lang?: Languages | SpecialLanguage;
    /**
     * A map of color names to themes.
     *
     * `light` and `dark` are required, and arbitrary color names can be added.
     *
     * @example
     * ```ts
     * themes: {
     *   light: 'vitesse-light',
     *   dark: 'vitesse-dark',
     *   soft: 'nord',
     *   // custom colors
     * }
     * ```
     */
    themes: Partial<Record<string, Themes | ThemeRegistrationAny | SpecialTheme>>;
}
interface ThemedTokenScopeExplanation {
    scopeName: string;
    themeMatches: any[];
}
interface ThemedTokenExplanation {
    content: string;
    scopes: ThemedTokenScopeExplanation[];
}
/**
 * A single token with color, and optionally with explanation.
 *
 * For example:
 *
 * ```json
 * {
 *   "content": "shiki",
 *   "color": "#D8DEE9",
 *   "explanation": [
 *     {
 *       "content": "shiki",
 *       "scopes": [
 *         {
 *           "scopeName": "source.js",
 *           "themeMatches": []
 *         },
 *         {
 *           "scopeName": "meta.objectliteral.js",
 *           "themeMatches": []
 *         },
 *         {
 *           "scopeName": "meta.object.member.js",
 *           "themeMatches": []
 *         },
 *         {
 *           "scopeName": "meta.array.literal.js",
 *           "themeMatches": []
 *         },
 *         {
 *           "scopeName": "variable.other.object.js",
 *           "themeMatches": [
 *             {
 *               "name": "Variable",
 *               "scope": "variable.other",
 *               "settings": {
 *                 "foreground": "#D8DEE9"
 *               }
 *             },
 *             {
 *               "name": "[JavaScript] Variable Other Object",
 *               "scope": "source.js variable.other.object",
 *               "settings": {
 *                 "foreground": "#D8DEE9"
 *               }
 *             }
 *           ]
 *         }
 *       ]
 *     }
 *   ]
 * }
 * ```
 */
interface ThemedToken extends TokenStyles, TokenBase {
}
interface TokenBase {
    /**
     * The content of the token
     */
    content: string;
    /**
     * The start offset of the token, relative to the input code. 0-indexed.
     */
    offset: number;
    /**
     * Explanation of
     *
     * - token text's matching scopes
     * - reason that token text is given a color (one matching scope matches a rule (scope -> color) in the theme)
     */
    explanation?: ThemedTokenExplanation[];
}
interface TokenStyles {
    /**
     * 6 or 8 digit hex code representation of the token's color
     */
    color?: string;
    /**
     * 6 or 8 digit hex code representation of the token's background color
     */
    bgColor?: string;
    /**
     * Font style of token. Can be None/Italic/Bold/Underline
     */
    fontStyle?: FontStyle;
    /**
     * Override with custom inline style for HTML renderer.
     * When specified, `color` and `fontStyle` will be ignored.
     */
    htmlStyle?: string;
}
interface ThemedTokenWithVariants extends TokenBase {
    /**
     * An object of color name to token styles
     */
    variants: Record<string, TokenStyles>;
}
interface TokenizeWithThemeOptions {
    /**
     * Include explanation of why a token is given a color.
     *
     * @default false
     */
    includeExplanation?: boolean;
    /**
     * A map of color names to new color values.
     *
     * The color key starts with '#' and should be lowercased.
     *
     * This will be merged with theme's `colorReplacements` if any.
     */
    colorReplacements?: Record<string, string | Record<string, string>>;
    /**
     * Lines above this length will not be tokenized for performance reasons.
     *
     * @default 0 (no limit)
     */
    tokenizeMaxLineLength?: number;
    /**
     * Time limit in milliseconds for tokenizing a single line.
     *
     * @default 500 (0.5s)
     */
    tokenizeTimeLimit?: number;
}
/**
 * Result of `codeToTokens`, an object with 2D array of tokens and meta info like background and foreground color.
 */
interface TokensResult {
    /**
     * 2D array of tokens, first dimension is lines, second dimension is tokens in a line.
     */
    tokens: ThemedToken[][];
    /**
     * Foreground color of the code.
     */
    fg?: string;
    /**
     * Background color of the code.
     */
    bg?: string;
    /**
     * A string representation of themes applied to the token.
     */
    themeName?: string;
    /**
     * Custom style string to be applied to the root `<pre>` element.
     * When specified, `fg` and `bg` will be ignored.
     */
    rootStyle?: string;
}
declare enum FontStyle {
    NotSet = -1,
    None = 0,
    Italic = 1,
    Bold = 2,
    Underline = 4
}

export { type ThemeRegistration as $, type RootContent as A, type BundledHighlighterOptions as B, type CodeToHastOptions as C, type ShikiTransformer as D, type Element as E, FontStyle as F, type AnsiLanguage as G, type HighlighterCoreOptions as H, INITIAL as I, type ResolveBundleKey as J, type LanguageRegistration as K, type LanguageInput as L, type MaybeArray as M, type Nodes as N, type BundledLanguageInfo as O, type PlainTextLanguage as P, type DynamicImportLanguageRegistration as Q, Registry as R, type StateStack as S, Theme as T, type CodeOptionsSingleTheme as U, type CodeOptionsMultipleThemes as V, type CodeOptionsThemes as W, type CodeToHastOptionsCommon as X, type CodeOptionsMeta as Y, type CodeToHastRenderOptionsCommon as Z, type ThemeRegistrationRaw as _, type IRawTheme as a, type DynamicImportThemeRegistration as a0, type BundledThemeInfo as a1, type ThemedTokenScopeExplanation as a2, type ThemedTokenExplanation as a3, type TokenBase as a4, type TransformerOptions as a5, type ShikiTransformerContextMeta as a6, type ShikiTransformerContext as a7, type Awaitable as a8, type MaybeGetter as a9, type MaybeModule as aa, type StringLiteralUnion as ab, type DecorationOptions as ac, type DecorationItem as ad, type ResolvedDecorationItem as ae, type DecorationTransformType as af, type Offset as ag, type OffsetOrPosition as ah, type ResolvedPosition as ai, type IRawGrammar as b, type IGrammar as c, type IGrammarConfiguration as d, type IOnigLib as e, type RegistryOptions as f, type IRawThemeSetting as g, type ThemeInput as h, type Root as i, type CodeToTokensOptions as j, type TokensResult as k, type RequireKeys as l, type CodeToTokensBaseOptions as m, type ThemedToken as n, type CodeToTokensWithThemesOptions as o, type ThemedTokenWithVariants as p, type SpecialLanguage as q, type SpecialTheme as r, type ThemeRegistrationAny as s, type TokenizeWithThemeOptions as t, type TokenStyles as u, type Position as v, type ThemeRegistrationResolved as w, type ShikiTransformerContextCommon as x, type CodeToHastRenderOptions as y, type ShikiTransformerContextSource as z };
