"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLanguageServicePlugins = exports.getLanguagePlugins = void 0;
const node_1 = require("@volar/language-server/node");
const vscode_uri_1 = require("vscode-uri");
const core_1 = require("./core");
const svelte_js_1 = require("./core/svelte.js");
const vue_js_1 = require("./core/vue.js");
const importPackage_js_1 = require("./importPackage.js");
const utils_js_1 = require("./utils.js");
// Services
const volar_service_css_1 = require("volar-service-css");
const volar_service_emmet_1 = require("volar-service-emmet");
const volar_service_prettier_1 = require("volar-service-prettier");
const volar_service_typescript_twoslash_queries_1 = require("volar-service-typescript-twoslash-queries");
const astro_js_1 = require("./plugins/astro.js");
const html_js_1 = require("./plugins/html.js");
const index_js_1 = require("./plugins/typescript-addons/index.js");
const index_js_2 = require("./plugins/typescript/index.js");
function getLanguagePlugins(connection, ts, serviceEnv, tsconfig) {
    const languagePlugins = [
        (0, vue_js_1.getVueLanguageModule)(),
        (0, svelte_js_1.getSvelteLanguageModule)(),
    ];
    const rootPath = tsconfig
        ? tsconfig.split('/').slice(0, -1).join('/')
        : serviceEnv.typescript.uriToFileName(serviceEnv.workspaceFolder);
    const nearestPackageJson = ts.findConfigFile(rootPath, ts.sys.fileExists, 'package.json');
    const astroInstall = (0, utils_js_1.getAstroInstall)([rootPath], {
        nearestPackageJson: nearestPackageJson,
        readDirectory: ts.sys.readDirectory,
    });
    if (astroInstall === 'not-found') {
        connection.sendNotification(node_1.ShowMessageNotification.type, {
            message: `Couldn't find Astro in workspace "${rootPath}". Experience might be degraded. For the best experience, please make sure Astro is installed into your project and restart the language server.`,
            type: node_1.MessageType.Warning,
        });
    }
    languagePlugins.unshift((0, core_1.getLanguageModule)(typeof astroInstall === 'string' ? undefined : astroInstall, ts));
    return languagePlugins;
}
exports.getLanguagePlugins = getLanguagePlugins;
function getLanguageServicePlugins(connection, ts) {
    return [
        (0, html_js_1.create)(),
        (0, volar_service_css_1.create)(),
        (0, volar_service_emmet_1.create)(),
        ...(0, index_js_2.create)(ts),
        (0, volar_service_typescript_twoslash_queries_1.create)(ts),
        (0, index_js_1.create)(),
        (0, astro_js_1.create)(ts),
        getPrettierService(),
    ];
    function getPrettierService() {
        let prettier;
        let prettierPluginPath;
        let hasShownNotification = false;
        return (0, volar_service_prettier_1.create)((context) => {
            const workspaceUri = vscode_uri_1.URI.parse(context.env.workspaceFolder);
            if (workspaceUri.scheme === 'file') {
                prettier = (0, importPackage_js_1.importPrettier)(workspaceUri.fsPath);
                prettierPluginPath = (0, importPackage_js_1.getPrettierPluginPath)(workspaceUri.fsPath);
                if ((!prettier || !prettierPluginPath) && !hasShownNotification) {
                    connection.sendNotification(node_1.ShowMessageNotification.type, {
                        message: "Couldn't load `prettier` or `prettier-plugin-astro`. Formatting will not work. Please make sure those two packages are installed into your project and restart the language server.",
                        type: node_1.MessageType.Warning,
                    });
                    hasShownNotification = true;
                }
                return prettier;
            }
        }, {
            documentSelector: ['astro'],
            getFormattingOptions: async (prettierInstance, document, formatOptions, context) => {
                const documentUri = context.decodeEmbeddedDocumentUri(document.uri)?.[0] ?? document.uri;
                const filePath = vscode_uri_1.URI.parse(documentUri).fsPath;
                if (!filePath) {
                    return {};
                }
                let configOptions = null;
                try {
                    configOptions = await prettierInstance.resolveConfig(filePath, {
                        // This seems to be broken since Prettier 3, and it'll always use its cumbersome cache. Hopefully it works one day.
                        useCache: false,
                        editorconfig: true,
                    });
                }
                catch (e) {
                    connection.sendNotification(node_1.ShowMessageNotification.type, {
                        message: `Failed to load Prettier config.\n\nError:\n${e}`,
                        type: node_1.MessageType.Warning,
                    });
                    console.error('Failed to load Prettier config.', e);
                }
                const editorOptions = await context.env.getConfiguration?.('prettier', document.uri);
                // Return a config with the following cascade:
                // - Prettier config file should always win if it exists, if it doesn't:
                // - Prettier config from the VS Code extension is used, if it doesn't exist:
                // - Use the editor's basic configuration settings
                const resolvedConfig = {
                    filepath: filePath,
                    tabWidth: formatOptions.tabSize,
                    useTabs: !formatOptions.insertSpaces,
                    ...editorOptions,
                    ...configOptions,
                };
                return {
                    ...resolvedConfig,
                    plugins: [...(await getAstroPrettierPlugin()), ...(resolvedConfig.plugins ?? [])],
                    parser: 'astro',
                };
                async function getAstroPrettierPlugin() {
                    if (!prettier || !prettierPluginPath) {
                        return [];
                    }
                    const hasPluginLoadedAlready = (await prettier.getSupportInfo()).languages.some((l) => l.name === 'astro') ||
                        resolvedConfig.plugins?.includes('prettier-plugin-astro'); // getSupportInfo doesn't seems to work very well in Prettier 3 for plugins
                    return hasPluginLoadedAlready ? [] : [prettierPluginPath];
                }
            },
        });
    }
}
exports.getLanguageServicePlugins = getLanguageServicePlugins;
//# sourceMappingURL=languageServerPlugin.js.map